/* tune-local.c - 
 *
 * Copyright (C) 2007  Jochen Voss <voss@seehuhn.de>
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301 USA
 */

#include <time.h>
#include <math.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "fast-dm.h"
#include "measure.h"
#include "jvrandom.h"

static inline long
dgels (char TRANS, long M, long N, long NRHS, double *A, long LDA, double *B,
       long LDB, double *WORK, long LWORK)
{
  extern  void  dgels_ (const char *TRANSp, const long *Mp, const long *Np,
                        const long *NRHSp, double *A, const long *LDAp,
                        double *B, const long *LDBp, double *WORK,
                        const long *LWORKp, long *INFOp);
  long  info;
  dgels_ (&TRANS, &M, &N, &NRHS, A, &LDA, B, &LDB, WORK, &LWORK, &info);
  return info;
}

static inline double
dnrm2 (long N, const double *X, long INCX)
{
  extern double dnrm2_ (const long *Np, const double *X, const long *INCXp);
  return  dnrm2_ (&N, X, &INCX);
}

static inline double
ddot(long n, const double *dx, long incx, const double *dy, long incy)
{
  extern double ddot_(const long *np, const double *dx, const long *incxp,
		      const double *dy, const long *incyp);
  return ddot_(&n, dx, &incx, dy, &incy);
}

int
main (int argc, char **argv)
{
  double  data[7000], rhs[2000], param[7], error_target;
  int  k, i;
  
  init_noise (time (NULL));
  
  init_ds();
  error_target = pow(10, -3.0);
  set_precision(-log10(error_target));
  param[0] = log(TUNE_PDE_DT_MIN);
  param[1] = log(TUNE_PDE_DT_SCALE);
  param[2] = log(TUNE_PDE_DT_MAX);
  param[3] = log(TUNE_DZ);
  param[4] = log(TUNE_DV);
  param[5] = log(TUNE_DT0);
  param[6] = 1;
  for (k=0; k<10000; ++k) {
    int  d0;
    double  f1, f2, err1, cost1, dp[7], f6, err2, cost2, f7;
    
    /* output 1 */
    printf("\tTUNE_PDE_DT_MIN = %g;\n", exp(param[0]));
    printf("\tTUNE_PDE_DT_SCALE = %g;\n", exp(param[1]));
    printf("\tTUNE_PDE_DT_MAX = %g;\n", exp(param[2]));
    printf("\tTUNE_DZ = %g;\n", exp(param[3]));
    printf("\tTUNE_DV = %g;\n", exp(param[4]));
    printf("\tTUNE_DT0 = %g;\n", exp(param[5]));
    
    /* data acquisition */
    for (d0=0; d0<1000; ++d0) {
      double  err, cost, v0[7];
      
      memcpy(v0, param, 7*sizeof(double));
      for (i=0; i<6; ++i)  v0[i] += NOISE(0.02);
      doit(v0, &err, &cost);
      for (i=0; i<7; ++i)  data[1000*i+d0] = v0[i];
      rhs[d0] = err;
      rhs[d0+1000] = cost;
    }
    
    /* data analysis */
    {
      double  m0[7000], f0;
      long  lwork;
      double  *work;
      
      memcpy (m0, data, 7000*sizeof(double));
      dgels('N', 1000, 7, 2, m0, 1000, rhs, 1000, &f0, -1);
      lwork = f0+0.5;
      work = malloc(lwork*sizeof(double));
      dgels('N', 1000, 7, 2, m0, 1000, rhs, 1000, work, lwork);
      
      free(work);
    }
    f1 = dnrm2(6, rhs, 1);
    f2 = dnrm2(6, rhs+1000, 1);
    err1 = ddot(7, param, 1, rhs, 1);
    cost1 = ddot(7, param, 1, rhs+1000, 1);
    if (k%2 == 0) {
      double  f3, f4;
      
      f3 = log(error_target)-err1;
      if (f3>0) {
        memcpy(dp, rhs+1000, 6*sizeof(double));
      } else {
        memcpy(dp, rhs, 6*sizeof(double));
      }
      f4 = ddot(6, dp, 1, rhs, 1);
      for (i=0; i<6; ++i)  dp[i] *= f3/f4;
    } else {
      double  f5;
      
      f5 = ddot(6, rhs, 1, rhs+1000, 1);
      for (i=0; i<6; ++i)  dp[i] = rhs[i]*f5/(f1*f1);
      for (i=0; i<6; ++i)  dp[i] -= rhs[i+1000];
      for (i=0; i<6; ++i)  dp[i] *= 0.05;
    }
    f6 = dnrm2(6, dp, 1);
    if (f6 > 0.02)  for (i=0; i<6; ++i)  dp[i] *= 0.02/f6;
    dp[6] = 0;
    err2 = 0;
    for (i=0; i<7; ++i)  err2 += (param[i]+dp[i])*rhs[i];
    cost2 = 0;
    for (i=0; i<7; ++i)  cost2 += (param[i]+dp[i])*rhs[i+1000];
    
    /* output 2 */
    printf("\t/* %gm CYC, error=%g (from %d samples) */\n", exp(cost1),
           exp(err1), 1000);
    printf("\n");
    f7 = ddot(6, rhs, 1, rhs+1000, 1);
    printf("cos(phi) = %g\n", f7/(f2*f1));
    printf("\n");
    printf("param................err ......cost ........dp\n");
    printf("PDE_DT_MIN    %10.2f %10.2f %10.3f\n", rhs[0]/f1, rhs[1000]/f2,
           dp[0]);
    printf("PDE_DT_SCALE  %10.2f %10.2f %10.3f\n", rhs[1]/f1, rhs[1001]/f2,
           dp[1]);
    printf("PDE_DT_MAX    %10.2f %10.2f %10.3f\n", rhs[2]/f1, rhs[1002]/f2,
           dp[2]);
    printf("DZ            %10.2f %10.2f %10.3f\n", rhs[3]/f1, rhs[1003]/f2,
           dp[3]);
    printf("DV            %10.2f %10.2f %10.3f\n", rhs[4]/f1, rhs[1004]/f2,
           dp[4]);
    printf("DT0           %10.2f %10.2f %10.3f\n", rhs[5]/f1, rhs[1005]/f2,
           dp[5]);
    printf("\n");
    printf("\t/* %gm CYC, error=%g (estimated) */\n", exp(cost2), exp(err2));
    
    if (k%2 == 1) {
      char  s0[80];
      FILE  *fd;
      
      snprintf(s0, 80, "tune%+#.2g.dat", log10(error_target));
      fd = fopen(s0, "w");
      fprintf(fd, "\t/* %gm CYC, error=%g (from %d samples) */\n", exp(cost1),
              exp(err1), 1000);
      fprintf(fd, "\tTUNE_PDE_DT_MIN = %g;\n", exp(param[0]));
      fprintf(fd, "\tTUNE_PDE_DT_SCALE = %g;\n", exp(param[1]));
      fprintf(fd, "\tTUNE_PDE_DT_MAX = %g;\n", exp(param[2]));
      fprintf(fd, "\tTUNE_DZ = %g;\n", exp(param[3]));
      fprintf(fd, "\tTUNE_DV = %g;\n", exp(param[4]));
      fprintf(fd, "\tTUNE_DT0 = %g;\n", exp(param[5]));
      fclose(fd);
    }
    
    /* update */
    for (i=0; i<7; ++i)  param[i] += dp[i];
  }
  
  return 0;
}
