/* simplex.c - the downhill simplex method of Nelder and Mead
 *
 * Copyright (C) 2006  Jochen Voss, Andreas Voss.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301 USA.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <assert.h>

#include "fast-dm.h"


static void
move_point (int n, double p, const double *x, const double *y, double *z)
/* Move a point in direction of another point.
 * x, y and z must be vectors of length n.
 * The function reads x and y and stores x + p*(y-x) in z.
 * For p=2 this calculates the reflection of x at y.  */
{
	int  i;
	for (i=0; i<n; ++i)  z[i] = x[i] + p*(y[i]-x[i]);
}

static int
compare_points (const void *a, const void *b)
{
	const double *da = a;
	const double *db = b;
	if (*da<*db)  return -1;
	if (*da>*db)  return +1;
	return 0;
}

double
simplex (int n, double *x, const double *eps, double size_limit, void *data,
	 double (*fn)(const double *x, void *data))
/* Multidimensional minimisation of the function 'fn' using the
 * downhill simplex algorithm of Nelder and Mead.
 *
 * 'fn' is the function to minimise.  The first argument of 'fn' is a
 * vector x which consists of 'n' doubles.  With the 'data' pointer
 * additional information may be forwarded to 'fn'.  The input values
 * of 'x' define one vertex of the starting simplex.  For the other
 * vertices, one of the input values of x is moved by the according
 * value of 'eps'.  'size_limit' defines the precision of the search.
 * The algorithm is terminated as soon as all edges of the simplex are
 * smaller than 'size_limit'.  On return the vector 'x' contains the
 * best found solution, that is, the parameters with the smallest
 * result of 'fn'.
 */
{
	double *points;
	double *mid, *tmp1, *tmp2;
	double  res;
	int  i, j, k;

	assert (n >= 2);
  
	/* Allocate an array for the vertices of the simplex.
	 * Vertex k is stored in entries k*(n+1)+1, ..., k*(n+1)+n,
	 * the function value at this vertex is stored in entry k*(n+1).
	 * Vertices are stored in order of increasing function values.  */
	points = xnew (double, (n+1)*(n+1));
	mid = xnew (double, (n+1));
	tmp1 = xnew (double, (n+1));
	tmp2 = xnew (double, (n+1));

	/* initialise the simplex */
	for (j=0; j<=n; ++j) {
		for (i=0; i<n; ++i) {
			points[j*(n+1)+1+i] = x[i] + ((i+1)==j ? eps[i] : 0);
		}
		points[j*(n+1)] = fn (points+j*(n+1)+1, data);
	}
  
	for (k=0; ; ++k) {
		qsort (points, n+1, (n+1)*sizeof(double), compare_points);

		/* stopping criterion */
		if (k>1000)  break;
		{
			double  size = 0.0;
			for (i=0; i<n; ++i) {
				double  min, max;

				min = max = points[1+i];
				for (j=1; j<=n; ++j) {
					double  x = points [j*(n+1)+1+i];
					if (x < min) {
						min = x;
					} else if (x > max) {
						max = x;
					}
				}
				if (max-min > size)  size = max-min;
				if (size > size_limit)  break;
			}
			if (size < size_limit)  break;
		}

		/* calculate the reflection point */
		for (i=1; i<=n; ++i)  mid[i] = points[i];
		for (j=1; j<n; ++j) {
			for (i=1; i<=n; ++i)  mid[i] += points[j*(n+1)+i];
		}
		for (i=1; i<=n; ++i)  mid[i] /= n;

		/* try to reflect */
		move_point (n, 2, points+n*(n+1)+1, mid+1, tmp1+1);
		tmp1[0] = fn (tmp1+1, data);
		if (tmp1[0] < points[(n-1)*(n+1)]) {
			/* accept the reflection */
			if (tmp1[0] < points[(n-1)*(n+1)]) {
				move_point (n, 3, points+n*(n+1)+1, mid+1,
					    tmp2+1);
				tmp2[0] = fn (tmp2+1, data);
				if (tmp2[0] < tmp1[0]) {
					/* replace the worst point with the
					 * extended reflection */
					memcpy (points+n*(n+1), tmp2,
						(n+1)*sizeof(double));
				} else {
					/* replace the worst point with the
					 * reflection */
					memcpy (points+n*(n+1), tmp1,
						(n+1)*sizeof(double));
				}
			}
			continue;
		}

		/* try to contract */
		move_point (n, 0.5, points+n*(n+1)+1, mid+1, tmp1+1);
		tmp1[0] = fn (tmp1+1, data);
		if (tmp1[0] < points[n*(n+1)]) {
			/* replace the worst point with the contraction */
			memcpy (points+n*(n+1), tmp1, (n+1)*sizeof(double));
			continue;
		}

		/* if everything else fails, use the multiple contraction */
		for (j=1; j<=n; ++j) {
			/* move point j halfways towards the best point */
			move_point (n, 0.5, points+j*(n+1)+1, points+1,
				    points+j*(n+1)+1);
			points[j*(n+1)] = fn (points+j*(n+1)+1, data);
		}
	}
  
	memcpy (x, points+1, n*sizeof(double));
	res = points[0];

	xfree (tmp1);
	xfree (tmp2);
	xfree (mid);
	xfree (points);
  
	return  res;
}

/*
 * Local Variables:
 * c-file-style: "linux"
 * End:
 */
