/* measure.c - measure the effect of tuning parameter settings
 *
 * Copyright (C) 2006  Jochen Voss, Andreas Voss.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301 USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <math.h>
#include <float.h>

#include <papi.h>

#include "fast-dm.h"
#include "measure.h"


static const int  N = 500;

#define DS_COUNT 11

/**********************************************************************
 * read the reference data
 */

struct ds {
	double  para[p_count], z;
	double *G;
};

static void
read_params(const char *fname, double *para, double *z)
{
	FILE *fp;

	fp = fopen(fname, "r");
	if (! fp) {
		perror("cannot open parameter file");
		exit(1);
	}
	fscanf(fp, "a = %lf\n", para+p_a);
	fscanf(fp, "z = %lf\n", z);
	fscanf(fp, "v = %lf\n", para+p_v);
	fscanf(fp, "t0 = %lf\n", para+p_t0);
	fscanf(fp, "sz = %lf\n", para+p_sz);
	fscanf(fp, "sv = %lf\n", para+p_sv);
	fscanf(fp, "st0 = %lf\n", para+p_st0);
	fclose(fp);
}

static void
read_data(const char *fname, int n, double *data)
{
	FILE *fp;
	int  i;

	fp = fopen(fname, "r");
	if (! fp) {
		perror("cannot open target data file");
		exit(1);
	}
	for (i=0; i<n; ++i) {
		double  t, x;
		int  rc;

		rc = fscanf(fp, "%lf %lf", &t, &x);
		if (rc < 2) {
			fprintf(stderr, "corrupted target data file\n");
			exit(1);
		}
		data[i] = x;
	}
	fclose(fp);
}

static void
read_ds(struct ds *ds, int i)
{
	char buffer[80];

	snprintf(buffer, 80, "%d.info", i);
	read_params(buffer, ds->para, &ds->z);

	ds->G = xnew(double, 2*N+1);
	snprintf(buffer, 80, "%d.dat", i);
	read_data(buffer, 2*N+1, ds->G);
}

/**********************************************************************
 * optmise the parameters
 */

FILE *logfile = NULL;

static void
eval_one(const struct ds *ds, double *T_ret, double *d_ret)
{
	int  counters [1] = { PAPI_TOT_CYC };
	double *F;
	struct F_calculator *fc;
	double  dist, dt = 5.0/N;
	long_long  elapsed;
	int  i;

	F = xnew(double, 2*N+1);

	PAPI_start_counters(counters, 1);
	fc = F_new(ds->para);
	F_start(fc, b_upper);
	for (i=0; i<=N; ++i) {
		F[N+i] = F_get_val(fc, i*dt, ds->z);
	}
	F_start(fc, b_lower);
	for (i=1; i<=N; ++i) {
		F[N-i] = F_get_val(fc, i*dt, ds->z);
	}
	F_delete(fc);
	PAPI_stop_counters(&elapsed, 1);
	*T_ret = elapsed / 1e6;

	dist = 0;
	for (i=0; i<2*N+1; ++i) {
		double  d = fabs(F[i]-ds->G[i]);
		if (d > dist)  dist = d;
	}
	*d_ret = dist;

	xfree(F);
}

static struct ds ds[DS_COUNT];

void
init_ds(void)
{
	int  i, nc;

	set_precision(4.0);
	for (i=0; i<DS_COUNT; ++i) {
		read_ds(ds+i, i+1);
	}

	nc = PAPI_num_counters();
	if (nc < 1) {
		fprintf(stderr,
			"not enough counters (need 1, have %d)\n", nc);
		exit(1);
	}
}

void
doit(const double *param, double *err_ret, double *cost_ret)
{
	double  elapsed, dist, d[DS_COUNT], e[DS_COUNT];
	int  i;

	TUNE_PDE_DT_MIN = exp(param[0]);
	TUNE_PDE_DT_SCALE = exp(param[1]);
	TUNE_PDE_DT_MAX = exp(param[2]);
	TUNE_DZ = exp(param[3]);
	TUNE_DV = exp(param[4]);
	TUNE_DT0 = exp(param[5]);

	for (i=0; i<DS_COUNT; ++i)
		eval_one(ds+i, e+i, d+i);
	elapsed = 0;
	dist = 0;
	for (i=0; i<DS_COUNT; ++i) {
		elapsed += e[i];
		if (d[i] > dist) dist = d[i];
	}

	*err_ret = log(dist);
	*cost_ret = log(elapsed);
}

/*
 * Local Variables:
 * c-file-style: "linux"
 * End:
 */
