#! /usr/bin/env python
# tune-local.py - autogenerate tune-local.c using CSimGen
#
# copyright (c) 2007  Jochen Voss  <voss@seehuhn.de>
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License as
# published by the Free Software Foundation; either version 2 of the
# License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
# 02110-1301 USA

from csimgen.value import value
from csimgen.code import Source
from csimgen.vector import ConstVector, ExplicitVector

source = Source("tune-local.c",
                license="GPL2")
source.add_include('"fast-dm.h"')
source.add_include('"measure.h"')
main = source.main()
main.init_noise()

main.funcall("init_ds")

N = 1000
data = main.define_matrix(N, 7, name="data", transposed=True)
rhs = main.define_matrix(N, 2, name="rhs", transposed=True)
param = main.define_vector(7, name="param")
error_target = main.define_double(name="error_target", init=main.pow10(-3.0))

main.funcall("set_precision", -main.log10(error_target))
main.store_double(param[0], main.log(value("TUNE_PDE_DT_MIN")))
main.store_double(param[1], main.log(value("TUNE_PDE_DT_SCALE")))
main.store_double(param[2], main.log(value("TUNE_PDE_DT_MAX")))
main.store_double(param[3], main.log(value("TUNE_DZ")))
main.store_double(param[4], main.log(value("TUNE_DV")))
main.store_double(param[5], main.log(value("TUNE_DT0")))
main.store_double(param[6], 1)

######################################################################

k = main.define_int(name="k")
mainloop = main.loop(k, 0, 10000)

mainloop.statement("/* output 1 */")
labels = [
    "PDE_DT_MIN", "PDE_DT_SCALE", "PDE_DT_MAX", "DZ", "DV", "DT0"
    ]
for i,label in enumerate(labels):
    mainloop.output_string(r'"\tTUNE_%s = %%g;\n"'%label, mainloop.exp(param[i]))

######################################################################

mainloop.statement("",
                   "/* data acquisition */")
i = mainloop.define_int()
loop = mainloop.loop(i, 0, N)
err = loop.define_double(name="err")
cost = loop.define_double(name="cost")
noise = ConstVector(6, "NOISE(0.02)")
p = loop.define_vector(7, init=param)
loop.iadd_vector(p[0:6], noise)
loop.funcall("doit", p, err.reference(), cost.reference())
drow = data.get_row(i)
loop.store_vector(drow, p)
loop.store_double(rhs[i,0], err)
loop.store_double(rhs[i,1], cost)

######################################################################

mainloop.statement("",
                   "/* data analysis */")
mainloop.least_squares_matrix_vector(data, rhs)

err = rhs.get_column(0)[0:6]
err_l = mainloop.norm2_vector(err)

cost = rhs.get_column(1)[0:6]
cost_l = mainloop.norm2_vector(cost)

err1 = mainloop.dot_vector_vector(param, rhs.get_column(0)[0:7],
                                  name="err1")
cost1 = mainloop.dot_vector_vector(param, rhs.get_column(1)[0:7],
                                  name="cost1")

dp = mainloop.define_vector(7, name="dp")
case1,case2 = mainloop.ifelse("%s%%2 == 0"%k)

# case 1: fix precision
delta = case1.define_double(init=case1.log(error_target)-err1);
c1,c2 = case1.ifelse("%s>0"%delta)
c1.store_vector(dp[0:6], cost)
c2.store_vector(dp[0:6], err)
case1.imult_vector(dp[0:6],
                   delta/case1.dot_vector_vector(dp[0:6],rhs.get_column(0)[0:6]))

# case 2: improve cost
q = case2.dot_vector_vector(err, cost)
case2.store_vector(dp[0:6], err*q/(err_l*err_l))
case2.isub_vector(dp[0:6], cost)
case2.imult_vector(dp[0:6], 0.05)

dp_l = mainloop.norm2_vector(dp[0:6])
c = mainloop.cond("%s > 0.02"%dp_l)
c.imult_vector(dp[0:6], 0.02/dp_l)
mainloop.store_double(dp[6], 0)

err2 = mainloop.dot_vector_vector(param+dp, rhs.get_column(0)[0:7],
                                  name="err2")
cost2 = mainloop.dot_vector_vector(param+dp, rhs.get_column(1)[0:7],
                                   name="cost2")

######################################################################

mainloop.statement("",
                   "/* output 2 */")
mainloop.output_string(r'"\t/* %gm CYC, error=%g (from %d samples) */\n"',
                       mainloop.exp(cost1), mainloop.exp(err1), N)
mainloop.output_string(r'"\n"')
mainloop.output_string(r'"cos(phi) = %g\n"',
                       mainloop.dot_vector_vector(err,cost)/(cost_l*err_l))

mainloop.output_string(r'"\n"')
mainloop.output_string(r'"param................err ......cost ........dp\n"')
for i,label in enumerate(labels):
    mainloop.output_string(r'"%-13s %%10.2f %%10.2f %%10.3f\n"'%label,
                           err[i]/err_l, cost[i]/cost_l, dp[i])

mainloop.output_string(r'"\n"')
mainloop.output_string(r'"\t/* %gm CYC, error=%g (estimated) */\n"',
                       mainloop.exp(cost2), mainloop.exp(err2))

mainloop.statement("")
c = mainloop.cond("%s%%2 == 1"%k)
fname = c.define_string(80, '"tune%+#.2g.dat"', c.log10(error_target))
fd = c.define_file("fd", fname=fname, fmode="w")
c.output_string(r'"\t/* %gm CYC, error=%g (from %d samples) */\n"',
                c.exp(cost1), c.exp(err1), N,
                fd=fd)
for i,label in enumerate(labels):
    c.output_string(r'"\tTUNE_%s = %%g;\n"'%label, c.exp(param[i]), fd=fd)
c.close_file(fd)

######################################################################

mainloop.statement("",
                   "/* update */")
mainloop.iadd_vector(param, dp)

source.save()
