/* tune-params.c - tune the adjustable parameters of fast-dm
 *
 * Copyright (C) 2006  Jochen Voss, Andreas Voss.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301 USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <math.h>
#include <float.h>
#include <signal.h>
#include <setjmp.h>
#include <sys/time.h>

#include <papi.h>

#include "fast-dm.h"


static const int  N = 500;

#define DS_COUNT 11

/**********************************************************************
 * read the reference data
 */

struct ds {
	double  para[p_count], z;
	double *G;
};

static void
read_params(const char *fname, double *para, double *z)
{
	FILE *fp;

	fp = fopen(fname, "r");
	if (! fp) {
		perror("cannot open parameter file");
		exit(1);
	}
	fscanf(fp, "a = %lf\n", para+p_a);
	fscanf(fp, "z = %lf\n", z);
	fscanf(fp, "v = %lf\n", para+p_v);
	fscanf(fp, "t0 = %lf\n", para+p_t0);
	fscanf(fp, "sz = %lf\n", para+p_sz);
	fscanf(fp, "sv = %lf\n", para+p_sv);
	fscanf(fp, "st0 = %lf\n", para+p_st0);
	fclose(fp);
}

static void
read_data(const char *fname, int n, double *data)
{
	FILE *fp;
	int  i;

	fp = fopen(fname, "r");
	if (! fp) {
		perror("cannot open target data file");
		exit(1);
	}
	for (i=0; i<n; ++i) {
		double  t, x;
		int  rc;

		rc = fscanf(fp, "%lf %lf", &t, &x);
		if (rc < 2) {
			fprintf(stderr, "corrupted target data file\n");
			exit(1);
		}
		data[i] = x;
	}
	fclose(fp);
}

static void
read_ds(struct ds *ds, int i)
{
	char buffer[80];

	snprintf(buffer, 80, "%d.info", i);
	read_params(buffer, ds->para, &ds->z);

	ds->G = xnew(double, 2*N+1);
	snprintf(buffer, 80, "%d.dat", i);
	read_data(buffer, 2*N+1, ds->G);
}

/**********************************************************************
 * optmise the parameters
 */

FILE *logfile = NULL;

static void
write_params(double elapsed, double dist, double accuracy)
{
	char buffer[80];
	static  double  best_elapsed = DBL_MAX;
	static  double  best_dist = DBL_MAX;
	FILE *fd;

	printf("%g %g %g %g %g %g -> %g*%g %g\n", TUNE_PDE_DT_MIN,
	       TUNE_PDE_DT_MAX, TUNE_PDE_DT_SCALE, TUNE_DZ, TUNE_DV,
	       TUNE_DT0, dist/accuracy, accuracy, elapsed);
	if (logfile) {
		fprintf(logfile, "%g %g %g %g %g %g %g %g\n",
			TUNE_PDE_DT_MIN, TUNE_PDE_DT_MAX,
			TUNE_PDE_DT_SCALE, TUNE_DZ, TUNE_DV,
			TUNE_DT0, dist, elapsed);
		fflush(logfile);
	}

	if (best_dist <= accuracy) {
		/* find the fastest among the valid parameter sets */
		if (dist > accuracy || elapsed >= best_elapsed) return;
	} else {
		/* try to get into the valid parameter range */
		if (dist > best_dist)  return;
	}

	snprintf(buffer, 80, "tune-%#.2g.dat", -log10(accuracy));
	fd = fopen(buffer, "w");
	fprintf(fd, "\t/* %gm CYC, error <= %g */\n", elapsed, dist);
	fprintf(fd, "\tTUNE_PDE_DT_MIN = %g;\n", TUNE_PDE_DT_MIN);
	fprintf(fd, "\tTUNE_PDE_DT_MAX = %g;\n", TUNE_PDE_DT_MAX);
	fprintf(fd, "\tTUNE_PDE_DT_SCALE = %g;\n", TUNE_PDE_DT_SCALE);
	fprintf(fd, "\tTUNE_DZ = %g;\n", TUNE_DZ);
	fprintf(fd, "\tTUNE_DV = %g;\n", TUNE_DV);
	fprintf(fd, "\tTUNE_DT0 = %g;\n", TUNE_DT0);
	fclose(fd);

	best_elapsed = elapsed;
	best_dist = dist;
}

static void
eval_one(const struct ds *ds, double *T_ret, double *d_ret)
{
	int  counters [1] = { PAPI_TOT_CYC };
	double *F;
	struct F_calculator *fc;
	double  dist, dt = 5.0/N;
	long_long  elapsed;
	int  i;

	F = xnew(double, 2*N+1);

	PAPI_start_counters(counters, 1);
	fc = F_new(ds->para);
	F_start(fc, b_upper);
	for (i=0; i<=N; ++i) {
		F[N+i] = F_get_val(fc, i*dt, ds->z);
	}
	F_start(fc, b_lower);
	for (i=1; i<=N; ++i) {
		F[N-i] = F_get_val(fc, i*dt, ds->z);
	}
	F_delete(fc);
	PAPI_stop_counters(&elapsed, 1);
	*T_ret = elapsed / 1e6;

	dist = 0;
	for (i=0; i<2*N+1; ++i) {
		double  d = fabs(F[i]-ds->G[i]);
		if (d > dist)  dist = d;
	}
	*d_ret = dist;

	xfree(F);
}

struct parameters {
	const struct ds *ds;
	double  accuracy;
};

static void
eval_all(const struct parameters *p, double *e_ret, double *d_ret)
{
	double  elapsed, dist, d[DS_COUNT], e[DS_COUNT];
	int  i;

	for (i=0; i<DS_COUNT; ++i)
		eval_one(p->ds+i, e+i, d+i);
	elapsed = 0;
	dist = 0;
	for (i=0; i<DS_COUNT; ++i) {
		elapsed += e[i];
		dist += pow(d[i],5.0);
	}
	dist = pow(dist, 1/5.0);

	*e_ret = elapsed;
	*d_ret = dist;
}

static void
minimiser2(const double *x, double res[2], void *data)
{
	struct parameters *p = data;
	double  elapsed, dist;

	TUNE_PDE_DT_MIN = exp(x[0]);
	TUNE_PDE_DT_MAX = exp(x[1]);
	TUNE_PDE_DT_SCALE = exp(x[2]);
	TUNE_DZ = exp(x[3]);
	TUNE_DV = exp(x[4]);
	TUNE_DT0 = exp(x[5]);

	eval_all(p, &elapsed, &dist);
	write_params(elapsed, dist, p->accuracy);
	if (dist > p->accuracy)
		res[0] = dist / p->accuracy;
	else
		res[0] = 0;
	res[1] = elapsed;
}

static  sigjmp_buf  leave_loop;

static void
handle_alarm(int signum)
{
	siglongjmp(leave_loop, 1);
}

int
main()
{
	struct ds ds[DS_COUNT];
	struct parameters  p;
	double  tune[6], tuneps[6];
	int  nc, i;

	for (i=0; i<DS_COUNT; ++i) {
		read_ds(ds+i, i+1);
	}
	p.ds = ds;

	nc = PAPI_num_counters();
	if (nc < 1) {
		fprintf(stderr,
			"not enough counters (need 1, have %d)\n", nc);
		exit(1);
	}

#if 1
	logfile = fopen("tune.log", "w");
	set_precision(4.2);

	/* 25873.2m CYC, error <= 7.91969e-05 */
	TUNE_PDE_DT_MIN = 0.000119998;
	TUNE_PDE_DT_MAX = 0.00515105;
	TUNE_PDE_DT_SCALE = 0.0121255;
	TUNE_DZ = 0.00347718;
	TUNE_DV = 0.00180531;
	TUNE_DT0 = 0.000133293;

	for (p.accuracy=pow(10,-4.1); p.accuracy<=0.1; p.accuracy/=pow(0.1,1/10.0)) {
		struct itimerval  pause;

		pause.it_interval.tv_usec = 0;
		pause.it_interval.tv_sec = 0;
		pause.it_value.tv_usec = 0;
		pause.it_value.tv_sec = 3600;

		signal(SIGVTALRM, handle_alarm);
		tune[0] = log(TUNE_PDE_DT_MIN);
		tune[1] = log(TUNE_PDE_DT_MAX);
		tune[2] = log(TUNE_PDE_DT_SCALE);
		tune[3] = log(TUNE_DZ);
		tune[4] = log(1.05*TUNE_DV);
		tune[5] = log(TUNE_DT0);
		tuneps[0] = tuneps[1] = tuneps[2] = +0.5;
		tuneps[3] = tuneps[4] = tuneps[5] = +0.5;
		if (sigsetjmp(leave_loop, 1) == 0) {
			if (setitimer(ITIMER_VIRTUAL, &pause, NULL) < 0)
				abort();
			simplex2(6, tuneps, 1e-4, minimiser2, tune, NULL, &p);
			putchar('\n');
			simplex2(6, tuneps, 1e-5, minimiser2, tune, NULL, &p);
			putchar('\n');
			simplex2(6, tuneps, 1e-6, minimiser2, tune, NULL, &p);
			putchar('\n');
			simplex2(6, tuneps, 1e-7, minimiser2, tune, NULL, &p);
		}
		puts("----------------------------------------");
	}
	fclose(logfile);
#else
	for (p.accuracy=0.1; ; p.accuracy*=pow(0.1,1/5.0)) {
		double  elapsed, dist;

		set_precision(-log10(p.accuracy));
		eval_all(&p, &elapsed, &dist);
		printf("%g %g %g\n", p.accuracy, dist, elapsed);
	}
#endif

	return  0;
}

/*
 * Local Variables:
 * c-file-style: "linux"
 * End:
 */
